/************************************************/
/* Demo Program For ET-BASE MEGA1280/2560 Board */
/************************************************/
/* Hardware    : ET-BASE MEGA1280/2560          */
/*             : ET-MINI SPI CAN CONTROL V1.0   */
/* MCU         : ATMEGA1280/2560                */
/*             : Run X-TAL 16.00 MHz            */
/* Text Editor : AVR Studio V4.18(Build 700)    */
/* Compiler    : WinAVR 2010-01-10  	        */
/* Create By   : ETT.CO.,LTD                    */
/* Last Update : June/2011                      */
/* Function    : Demo SPI CAN MCP2515 Interface */
/*             : Example Selftest Normal Send   */
/************************************************/
// ATMEGA1280    <-> ET-MINI SPI CAN CONTROL V1.0
// PB4(GPIO:OUT) --> CAN_CS
// PB3(SPI:MISO) <-- CAN_MISO
// PB2(SPI:MOSI) --> CAN_MOSI
// PB1(SPI:SCK)  --> CAN_SCK

#include <avr/io.h>
#include <stdio.h>
#include "mcp2515.h"												//MCP2515 Hardware Interface

#define F_CPU 16000000UL  											//16.0 MHz
#include <util/delay.h>
#define BAUD 9600													//9600 BPS
#define MYUBRR F_CPU/16/BAUD-1										//USART Normal Speed

//CAN Interface
#define		CAN_RTR		0x80									
#define		CAN_EID		0x40
#define		CAN_SID		0x00

//LED Status
#define 	PORT_LED 	PORTG										//Port Drive LED = PG
#define 	DIR_LED  	DDRG										//Port Direction	
#define 	LED 5               									//Pin Drive LED = PG5

void init_serial(unsigned int ubrr); 								//Initil UART
int	my_putchar(char c, FILE *stream);
int	my_getchar(FILE *stream);
void delay(uint16_t ms);
void can_send_message(uint32_t id, uint8_t *data, uint8_t length, uint8_t flags);

/* Retarget STDIO(putchar,getchar of printf) to My Function */
FILE uart_str = FDEV_SETUP_STREAM(my_putchar, my_getchar, _FDEV_SETUP_RW);

int main(void)
{
  uint8_t dat_can1[8] = { 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80 };
  uint8_t dat_can2[8] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37 };

  init_serial(MYUBRR);	   											//Initilial UART0 = 9600,N,8,1
  stdout = stdin = &uart_str;

  /* Configure LED port as output. */
  DIR_LED |= (1<<LED);												//Pin Drive LED = Out
	
  printf("\n\n\n\n");
  printf("=====================================================\n");
  printf(" ET-BASE MEGA1280/2560:Demo ET-MINI CAN CONTROL V1.0 \n"); 
  printf(" Demo Selftest Normal Mode -> Transmitt Frame        \n"); 
  printf("=====================================================\n"); 	
 

  /* Start Initial CAN Controller */

  //Speed 62.5Kbps -> Bus Length Max = 1000m.
  //Speed 125Kbps  -> Bus Length Max = 500m.
  //Speed 250Kbps  -> Bus Length Max = 250m.
  //Speed 500Kbps  -> Bus Length Max = 100m.
  //Speed 1Mbps    -> Bus Length Max = 30m.
  if (!mcp2515_init(CAN_500kbps)) 
  {
	printf("Initial MCP2515 Error\n");
	for (;;);
  }
  else 
  {
    printf("Initial CAN Speed 500Kbps\n");
    printf("Initial MCP2515 Complete\n");
  }

  /* End of Initial CAN Controller */

  
  while(1)
  {
    //Standard ID = 11 Bit Standard ID
	// 0000 0000 0000 0000 0000 0SSS SSSS SSSS 
	// Standard ID(0x0000..0x07FF)
    can_send_message( 0x00000123, dat_can1, 8, CAN_SID );		//Standard ID(0x0123)
    PORT_LED ^= (1<<LED);  										//Toggle Pin LED 
    delay(1000);

	// Extended ID = 29 Bit Address (11Bit Standard ID)+(18 Bit Extended ID)
	// 000S SSSS SSSS SSEE EEEE EEEE EEEE EEEE
	// Extended ID(0x0000..0x07FF + 0x000000..03FFFF)
	can_send_message( 0x1FFC0000, dat_can2, 8, CAN_EID );		//Extended ID(0x07FF + 0x000000)
	PORT_LED ^= (1<<LED);  										//Toggle Pin LED
    delay(1000);
  }
}

/****************/
/* Initial UART */
/****************/
void init_serial(unsigned int ubrr)  
{
  UBRR0H = (unsigned char)(ubrr>>8);							//Set baud rate 
  UBRR0L = (unsigned char)ubrr;
  UCSR0B = (1<<RXEN0)|(1<<TXEN0);								//Enable receiver and transmitter 
  UCSR0C = (1<<USBS0)|(3<<UCSZ00); 								//Set frame format: 8data, 2stop bit   
}

/*****************************/
/* Write 1 Character To UART */
/*****************************/
int my_putchar(char c, FILE *stream)
{
  if (c == '\n') 												//If printf = CR
  {
    my_putchar('\r', stream);									//Auto Insert LF
  }

  while (!(UCSR0A & (1<<UDRE0)));								//Wait TXD Buffer Empty   
  UDR0 = c;

  return 0;
}

/***************************/
/* Get character From UART */
/***************************/
int my_getchar(FILE *stream)  
{
  while(!(UCSR0A & (1<<RXC0)));									//Wait RXD Receive Data Ready
  return (UDR0);												//Get Receice Data & Return
}

/************/
/* Delay mS */
/************/
void delay(uint16_t ms)
{
  uint16_t ms_count;
  	
  while (ms) 
  {
    ms_count = F_CPU / 5000;
    while (ms_count) 
	{
	  asm volatile ("nop");
	  ms_count--;
    }
    ms--;
  }
}

/*
 *	Send Message CAN to MCP2515 (Buffer:0)
 *
 * Input 	: id		Identifier
 * 		 	: flags		Frame Type Flags (CAN_RTR oder CAN_EID)
 * 		 	: length	Length of Data Byte
 * 		 	: *data		Pointer of Data Byte
 * Output 	: none
 */
void can_send_message(uint32_t id, uint8_t *data, uint8_t length, uint8_t flags)
{
  uint8_t temp[4];
	
  if(length > 8)	//Over Length Buffer
  {
  	length = 0;
  }
	
  // Mask TX:0 High Proirity Message
  mcp2515_bit_modify(TXB0CTRL,(1<<TXP1)|(1<<TXP0), (1<<TXP1)|(1<<TXP0));
	
  //temp[0]=TXBnSIDH[XXXXXXXX] -> Standard ID[10..3]
  //temp[1]=TXBnSIDL[XXX-X-XX] -> Standard ID[2..0],EXIDE,EID17,EID16
  //temp[2]=TXBnEID8[XXXXXXXX] -> Extended ID[15..8]
  //temp[3]=TXBnEID0[XXXXXXXX] -> Extended ID[7..0]

  // Verify If Extend Frame
  if(flags & CAN_EID) 
  {
    temp[0] = (uint8_t) (id>>21);				
	temp[1] = (((uint8_t) (id>>16)) & 0x03) | (((uint8_t) (id>>13)) & 0xE0) | (1<<EXIDE);
	temp[2] = (uint8_t) (id>>8);				
	temp[3] = (uint8_t) id;					
  }

  //If Standard Frame
  else 
  {
    temp[0] = (uint8_t) (id>>3);				
	temp[1] = (uint8_t) (id<<5);				
	temp[2] = 0;    							
	temp[3] = 0;								
  }
  mcp2515_write_register_p(TXB0SIDH,temp, 4);					//TXBnSIDH,TXBnSIDL,TXBnEID8,TXBnEID0
		
  // Verify Frame = Remote Transmission Request
  if (flags & CAN_RTR)
  {
	mcp2515_write_register(TXB0DLC, length|(1<<RTR));
  }
  else
  {
    mcp2515_write_register(TXB0DLC, length);
  }
	
  // Data Byte of Frame
  mcp2515_write_register_p(TXB0D0,data,length );
	
  //Send RTS Instruction
  MCP2515_CS_LOW();												//Enable CS#(CS = Low)
  spi_putc(SPI_RTS|0x01);										//Start RTS0 
  MCP2515_CS_SET();												//Disable CS#(CS = High)
}

